/* FILE: mqueue.c                               (D. Tottingham  07/31/89)

This is a collection of C functions that manage the data queues for xdetect.
All functions have been written and compiled medium model. The following
functions are included:

q_delete_link ()                delete a data link from a data queue
q_dequeue ()                    dequeue a data link from a data queue
q_enqueue ()                    enqueue a data link on a data queue
q_initialize ()                 initialize a data queue
q_insert_link ()                insert a data link into a data queue

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <malloc.h>
#include <stdio.h>

#include "mconst.h"
#include "merror.h"
#include "mqueue.h"


/*=======================================================================*
 *                               dequeue                                 *
 *=======================================================================*/
/* Dequeue a data link from the data queue.                              */

PRIVATE
FLAG dequeue ( queue_ptr, link_ptr_ptr )
Q_QUEUE * queue_ptr;
Q_LINK * * link_ptr_ptr;
{
   if (queue_ptr->head != NULL) {
      *link_ptr_ptr = queue_ptr->head;
      queue_ptr->head = queue_ptr->head->next;

      if (queue_ptr->head == NULL)
         queue_ptr->tail = NULL;
      return (TRUE);

   } else return (FALSE);
}

/*=======================================================================*
 *                               enqueue                                 *
 *=======================================================================*/
/* Enqueue a data link to the data queue.                                */

PRIVATE
void enqueue ( queue_ptr, link_ptr )
Q_QUEUE * queue_ptr;
Q_LINK * link_ptr;
{
   if (queue_ptr->tail == NULL) {
      /* First link */
      queue_ptr->tail = link_ptr;
      queue_ptr->head = link_ptr;
   } else {
      queue_ptr->tail->next = link_ptr;
      queue_ptr->tail = link_ptr;
   }
}

/*=======================================================================*
 *                            q_delete_link                              *
 *=======================================================================*/
/* Delete a data link from a data queue.                                 */

PUBLIC
FLAG q_delete_link (queue_ptr, current_ptr, previous_ptr, type_ptr)
Q_QUEUE * queue_ptr;
Q_LINK * current_ptr;
Q_LINK * previous_ptr;
Q_TYPE * type_ptr;
{
   Q_LINK * link_ptr;

   if (previous_ptr == NULL)
      return ( q_dequeue(queue_ptr, type_ptr));
   else {
      previous_ptr->next = current_ptr->next;
      *type_ptr = current_ptr->type;
      free (current_ptr);
      return (TRUE);
   }
}

/*=======================================================================*
 *                              q_dequeue                                *
 *=======================================================================*/
/* Delete a data link from a data queue.                                 */

PUBLIC
FLAG q_dequeue (queue_ptr, type_ptr)
Q_QUEUE * queue_ptr;
Q_TYPE * type_ptr;
{
   Q_LINK * link_ptr;

   if (dequeue (queue_ptr, &link_ptr)) {
      *type_ptr = link_ptr->type;
      free (link_ptr);
      return (TRUE);
   } else return (FALSE);
}

/*=======================================================================*
 *                              q_enqueue                                *
 *=======================================================================*/
/* Create and save a data link in a data queue.                          */

PUBLIC
void q_enqueue (queue_ptr, type)
Q_QUEUE * queue_ptr;
Q_TYPE type;
{
   Q_LINK * new_link_ptr;

   new_link_ptr = (Q_LINK *) calloc (1, sizeof(Q_LINK));

   if (new_link_ptr == NULL) er_abort (Q_NO_STORAGE);

   new_link_ptr->type = type;
   new_link_ptr->next = NULL;
   enqueue (queue_ptr, new_link_ptr);
}

/*=======================================================================*
 *                           q_initialize                                *
 *=======================================================================*/
/* Initialize a data queue.                                              */

PUBLIC
void q_initialize ( queue_ptr )
Q_QUEUE * queue_ptr;
{
   queue_ptr->head = NULL;
   queue_ptr->tail = NULL;
}

/*=======================================================================*
 *                            q_insert_link                              *
 *=======================================================================*/
/* Insert a data link into a data queue.                                 */

PUBLIC
void q_insert_link (queue_ptr, previous_ptr, type)
Q_QUEUE * queue_ptr;
Q_LINK * previous_ptr;
Q_TYPE type;
{
   Q_LINK * link_ptr;

   link_ptr = (Q_LINK *) calloc (1, sizeof(Q_LINK));
   if (link_ptr == NULL) er_abort (Q_NO_STORAGE);
   link_ptr->type = type;

   if (previous_ptr == NULL) {
      link_ptr->next = queue_ptr->head;
      queue_ptr->head = link_ptr;
   } else {
      link_ptr->next = previous_ptr->next;
      previous_ptr->next = link_ptr;
   }
}
